package com.example.sefinsa_app;

import android.content.ContentValues;
import android.content.Context;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;
import androidx.work.BackoffPolicy;
import androidx.work.Constraints;
import androidx.work.ExistingPeriodicWorkPolicy;
import androidx.work.ExistingWorkPolicy;
import androidx.work.ListenableWorker;
import androidx.work.NetworkType;
import androidx.work.OneTimeWorkRequest;
import androidx.work.Worker;
import androidx.work.WorkerParameters;

import com.android.volley.DefaultRetryPolicy;
import com.android.volley.Request;
import com.android.volley.toolbox.JsonObjectRequest;
import com.example.sefinsa_app.api.API;
import com.example.sefinsa_app.migrations.DatabaseHelper;
import com.example.sefinsa_app.models.Aval;
import com.google.gson.Gson;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import androidx.work.PeriodicWorkRequest;
import androidx.work.WorkManager;

public class AvalesWorker extends Worker {
    private SharedPreferences sesion;
    private final ConnectivityManager connectivityManager;
    private final boolean isWorkEnqueued = false; // Bandera para evitar encolar múltiples veces
    public static final int BATCH_SIZE_AVALES = 999999; // Tamaño del lote
    public static int currentPageAvales = 0; // Página actual
    public static boolean allDataLoadedAvales = false;
    public static boolean isTaskAvalesCompleted = false;
    private static final Object dbLock = new Object(); // Usar un objeto de bloqueo para la base de datos
    @RequiresApi(api = Build.VERSION_CODES.N)
    public AvalesWorker(Context context, WorkerParameters workerParams) {
        super(context, workerParams);
        sesion = context.getSharedPreferences("sesion", Context.MODE_PRIVATE);
        connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
    }

    @RequiresApi(api = Build.VERSION_CODES.N)
    @NonNull
    @Override
    public ListenableWorker.Result doWork() {

        SQLiteDatabase db = null;
        DatabaseHelper dbHelper = null;

        try {
            // Comprobamos la disponibilidad de red
            if (!isNetworkAvailable(getApplicationContext())) {
                Log.d("AvalesWorker", "No internet connection, retrying later.");
                return ListenableWorker.Result.failure(); // Si ocurre un error, marcar como fallo
            }

            // Abrir la base de datos antes de realizar operaciones
            dbHelper = new DatabaseHelper(getApplicationContext());
            db = dbHelper.getWritableDatabase();  // Abrimos la base de datos aquí

            Context context = getApplicationContext();
            if (context == null) {
                Log.e("AvalesWorker", "El contexto es nulo. Abortando ejecución.");
                return Result.failure();
            }

            if (db == null) {
                Log.e("AvalesWorker", "La base de datos es nula. Abortando ejecución.");
                return Result.failure();
            }

            // Verificar si la base de datos está abierta correctamente
            if (db != null && db.isOpen()) {
                Log.d("AvalesWorker", "Base de datos abierta correctamente desde worker AVALES.");
            } else {
                Log.e("AvalesWorker", "Fallo al abrir la base de datos.");
                return ListenableWorker.Result.failure();
            }

            // Realizar las operaciones necesarias, dependiendo de si los datos ya fueron cargados o no
            if (!allDataLoadedAvales) {
                loadAllData(db);  // Pasamos la base de datos para realizar las inserciones/consultas
            } else {
                checkForUpdates(getApplicationContext(), db);  // Pasamos la base de datos para actualizar los datos
            }
            reprogramarWorker();
            return ListenableWorker.Result.success();
        } catch (Exception e) {
            Log.e("AvalesWorker", "Error in doWork", e);  // En caso de error, se captura la excepción
            return ListenableWorker.Result.failure();  // Retorna failure si ocurrió un error
        }
    }
    private void reprogramarWorker() {
        WorkManager.getInstance(getApplicationContext()).enqueueUniqueWork(
                "AvalesWorker",
                ExistingWorkPolicy.REPLACE, // Evita que se creen trabajos duplicados
                new OneTimeWorkRequest.Builder(AvalesWorker.class)
                        .setInitialDelay(12, TimeUnit.MINUTES)
                        .build()
        );

        Log.d("AvalesWorker", "Worker AvalesWorker reprogramado para ejecutarse en 2 minuto.");
    }

    public static void enqueueWork(Context context) {
        if (context == null) {
            Log.e("AvalesWorker", "No se puede iniciar WorkManager con un contexto nulo.");
            return;
        }

        // Configurar restricciones para el trabajo
        Constraints constraints = new Constraints.Builder()
                .setRequiredNetworkType(NetworkType.CONNECTED) // Requiere conexión a Internet
                .build();

        // Crear un OneTimeWorkRequest para que el trabajo se ejecute una sola vez
        OneTimeWorkRequest oneTimeWorkRequest = new OneTimeWorkRequest.Builder(AvalesWorker.class)
                .setConstraints(constraints) // Aplicar las restricciones
                //.setBackoffCriteria(BackoffPolicy.EXPONENTIAL, 10, TimeUnit.MINUTES) // Configurar reintentos en caso de fallo
                .build();

        // Encolar el trabajo único
        WorkManager.getInstance(context.getApplicationContext()).enqueueUniqueWork(
                "AvalesWorker", // Identificador único para el trabajo
                ExistingWorkPolicy.REPLACE, // Reemplaza el trabajo si ya existe uno en cola con el mismo nombre
                oneTimeWorkRequest
        );

        Log.d("AvalesWorker", "Se ha encolado un OneTimeWorkRequest para AvalesWorker.");
    }
    @RequiresApi(api = Build.VERSION_CODES.N)
    private void loadAllData(SQLiteDatabase db) {
        Context context = getApplicationContext();

        if (context == null) {
            Log.e("AVALES", "El contexto es nulo. No se puede cargar la data.");
            return;  // Salir del método si el contexto es nulo
        }

        sesion = context.getSharedPreferences("sesion", Context.MODE_PRIVATE);

        JSONObject data = new JSONObject();
        try {
            data.put("page", currentPageAvales);
            data.put("size", BATCH_SIZE_AVALES);
            data.put("func", "indexApp_lotes");
            Log.d("Avales", "Parametros enviados WORKER AVALES indexApp_lotes: " + data);
        } catch (JSONException e) {
            e.printStackTrace();
            return;
        }

        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlAvales, data,
                response -> {
                        try {
                            JSONArray data1 = response.getJSONArray("data");
                            Log.d("AVALES", "Cantidad de datos recibidos: " + data1.length());
                            List<Aval> batchAvales = new ArrayList<>();

                            // Procesar cada objeto JSON
                            for (int i = 0; i < data1.length(); i++) {
                                JSONObject obj = data1.getJSONObject(i);
                                Gson gson = new Gson();
                                Aval aval = gson.fromJson(obj.toString(), Aval.class);
                                batchAvales.add(aval);
                            }

                            // Insertar los avales en SQLite en un solo lote
                            insertAvalesIntoSQLiteAllData(context, batchAvales, db);

                            // Verificar si se han cargado todos los datos
                            if (batchAvales.size() < BATCH_SIZE_AVALES) {
                                allDataLoadedAvales = true;
                            } else {
                                currentPageAvales++;
                            }
                        } catch (JSONException e) {
                            Log.e("AVALES", "Error al procesar la respuesta JSON: " + e.getMessage());
                        }
                }, error -> {
            // Manejo del error de red
            Log.e("AVALES", "Error de red: " + error.toString());

            // Verificar si el error incluye una respuesta
            if (error.networkResponse != null && error.networkResponse.data != null) {
                String responseBody = new String(error.networkResponse.data);
                Log.e("AVALES", "Respuesta de error del servidor: " + responseBody); // Log del HTML o el error exacto
            }
            handleNetworkError(context);
        }
        );

        // Configuración del retry policy para la solicitud
        request.setRetryPolicy(new DefaultRetryPolicy(
                30000,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));

        VolleySingleton.getInstance(context.getApplicationContext()).addToRequestQueue(request);
    }
    private void handleNetworkError(Context context) {
        if (isNetworkAvailable(context)) {
            Log.d("AvalesWorker", "Network available, resuming work.");
            if (!isWorkEnqueued) {
                enqueueWork(context);
            }
        } else {
            Log.d("AvalesWorker", "No network available, stopping work.");
            cancelAndRescheduleWork(context);
        }
    }

    private void cancelAndRescheduleWork(Context context) {
        WorkManager.getInstance(context).cancelAllWorkByTag("AvalesWorker");
        scheduleRetry(context);
    }

    private void scheduleRetry(Context context) {
        Constraints constraints = new Constraints.Builder()
                .setRequiredNetworkType(NetworkType.CONNECTED)
                .build();

        OneTimeWorkRequest retryWorkRequest = new OneTimeWorkRequest.Builder(AvalesWorker.class)
                .setConstraints(constraints)
                .setInitialDelay(12, TimeUnit.MINUTES) // Reprogramar después de 5 minutos
                .build();

        WorkManager.getInstance(context).enqueue(retryWorkRequest);
        Log.d("AvalesWorker", "Reprogramado trabajo debido a error.");
    }

    private void insertAvalesIntoSQLiteAllData(Context context, List<Aval> avales, SQLiteDatabase db) {
        //Log.e("SQLite", "Entrando a la función insertAvalesIntoSQLite()............................");
        if (!db.isOpen()) {
            DatabaseHelper dbHelper = new DatabaseHelper(context);
            db = dbHelper.getWritableDatabase();
        }
        // Usar una transacción para manejar la inserción en lote
        db.beginTransaction();
        try {
            for (Aval aval : avales) {
                // Comprobar si el aval ya existe en la base de datos
                Cursor cursor = db.query("avales", new String[]{"id"}, "id = ?", new String[]{aval.getId()}, null, null, null);
                if (cursor != null && cursor.moveToFirst()) {
                    // El aval ya existe, no hacer nada
                    //Log.d("SQLite", "El aval con ID " + aval.getId() + " ya existe. No se realiza ninguna inserción.");
                    cursor.close(); // Cerrar el cursor
                } else {
                    // El aval no existe, insertarlo
                    ContentValues values = new ContentValues();
                    values.put("id", aval.getId());
                    values.put("nombre_completo", aval.getNombre_completo());
                    values.put("direccion", aval.getDireccion());
                    values.put("telefono", aval.getTelefono());
                    values.put("garantias", aval.getGarantias());
                    values.put("carpeta_comprobantes", aval.getCarpeta_comprobantes());
                    values.put("carpeta_garantias", aval.getCarpeta_garantias());
                    values.put("otras_referencias", aval.getOtras_referencias());
                    values.put("created_at", aval.getCreated_at());
                    values.put("updated_at", aval.getUpdated_at());
                    values.put("update_comprobantes", aval.getUpdate_comprobantes_Enaval());
                    values.put("update_garantias", aval.getUpdate_garantias_Enaval());
                    values.put("ruta_id", aval.getRuta_id() != null ? aval.getRuta_id() : "NULL");
                    values.put("poblacion_id", aval.getPoblacion_id() != null ? aval.getPoblacion_id() : "NULL");
                    values.put("colocadora_id", aval.getColocadora_id() != null ? aval.getColocadora_id() : "NULL");

                    long newRowId = db.insert("avales", null, values);
                    if (newRowId == -1) {
                        //Log.e("SQLite", "Error al insertar el aval con ID: " + aval.getId());
                    }
                }
            }
            db.setTransactionSuccessful(); // Marcar la transacción como exitosa
        } catch (Exception e) {
            Log.e("SQLite", "Error durante la inserción en lote: " + e.getMessage());
        } finally {
            if (db != null && db.isOpen()) {
                Log.e("AvalesWorker", "Cerrando la conexión a la base de datos insertAvalesIntoSQLiteAllData...");
                db.endTransaction();
                db.close();
                isTaskAvalesCompleted=true;
            }
        }
    }

    private void checkForUpdates(Context context, SQLiteDatabase db) {
        Log.d("AvalesWorker", "Checking for updates Avales...");

        Handler mainHandler = new Handler(Looper.getMainLooper());

        Gson gson = new Gson();

        DatabaseHelper dbHelper = new DatabaseHelper(context);

        // Obtener los últimos 40 pagos desde SQLite
        List<Aval> localAvales = dbHelper.getLast30AvalesFromSQLite();
        // Crear el objeto JSON para los parámetros de la solicitud
        JSONObject data = new JSONObject();
        try {
            data.put("func", "indexApp_Upd");
            data.put("ruta_id", sesion.getString("rutas", ""));
            //Log.d("Avales", "Parametros enviados WORKER indexApp_Upd: " + data.toString());
        } catch (JSONException e) {
            e.printStackTrace();
            return; // Salir si hay un error al crear el JSON
        }

        // Si el contexto no es nulo, realizar la solicitud HTTP
        if (context != null) {
            // Mover la solicitud HTTP y procesamiento al hilo secundario usando Executor
                JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlAvales, data,
                        response -> {
                                try {

                                    JSONArray data1 = response.getJSONArray("data");
                                    List<Aval> newAvales = new ArrayList<>();

                                    for (int i = 0; i < data1.length(); i++) {
                                        JSONObject obj = data1.getJSONObject(i);
                                        Aval serverData = gson.fromJson(obj.toString(), Aval.class);
                                        newAvales.add(serverData);
                                    }
                                    updateAvalInSQLite(context, newAvales, localAvales, db);
                                } catch (JSONException e) {
                                    Log.e("AVALES", "Error al procesar la respuesta JSON: " + e.getMessage());
                                }
                        },
                        error -> {
                            // Manejar errores solo si hay contexto
                            mainHandler.post(() -> {
                                if (context != null) {
                                    Log.e("AvalesWorker", "Error occurred: " + error.getMessage(), error);
                                } else {
                                    Log.e("AvalesWorker", "Context is null, cannot handle network error.");
                                }
                            });
                        }
                );

                // Configurar el retry policy
                request.setRetryPolicy(new DefaultRetryPolicy(
                        10000,
                        DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                        DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));

                // Agregar la solicitud a la cola de Volley
                VolleySingleton.getInstance(context.getApplicationContext()).addToRequestQueue(request);
        } else {
            // Si no hay contexto, solo trabajar con la caché
            Log.d("AvalesWorker", "No context available, skipping network requests.");
        }
    }
    private void updateAvalInSQLite(Context context, List<Aval> serverAvales, List<Aval> localAvales, SQLiteDatabase db) {
        if (!db.isOpen()) {
            Log.d("AvalesWorker", "Base de datos cerrada, reabriendo...");
            DatabaseHelper dbHelper = new DatabaseHelper(context);
            db = dbHelper.getWritableDatabase();
        }

        // Crear un conjunto con los IDs de los avales locales
        Set<String> localAvalIds = new HashSet<>();
        for (Aval localAval : localAvales) {
            localAvalIds.add(localAval.getId());
        }

        Log.d("AvalesWorker", "Iniciando actualización de avales...");

        // Inicializar contador
        AtomicInteger remainingComparisons = new AtomicInteger(serverAvales.size());

        // Iterar sobre los avales del servidor
        for (Aval serverAval : serverAvales) {
            Log.d("AvalesWorker", "Procesando aval ID: " + serverAval.getId());

            if (localAvalIds.contains(serverAval.getId())) {
                Log.d("AvalesWorker", "El aval ya existe en la base de datos local. Verificando actualización...");

                // Buscar el aval local correspondiente
                Aval localAval = null;
                for (Aval aval : localAvales) {
                    if (aval.getId().equals(serverAval.getId())) {
                        localAval = aval;
                        break;
                    }
                }

                if (localAval != null) {
                    String localUpdatedAt = localAval.getUpdated_at();

                    // Si localUpdatedAt es null o vacío, asignar la hora actual
                    if (localUpdatedAt == null || localUpdatedAt.isEmpty()) {
                        Log.d("AvalesWorker", "El campo 'updated_at' es nulo o vacío. Asignando nueva hora...");
                        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.O) {
                            LocalDateTime now = LocalDateTime.now();
                            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss", new Locale("es", "MX"));
                            localUpdatedAt = now.format(formatter);
                        } else {
                            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", new Locale("es", "MX"));
                            localUpdatedAt = sdf.format(new Date());
                        }

                        // Insertar la nueva fecha en la base de datos
                        ContentValues values = new ContentValues();
                        values.put("updated_at", localUpdatedAt);
                        int rowsUpdated = db.update("avales", values, "id = ?", new String[]{localAval.getId()});

                        if (rowsUpdated > 0) {
                            Log.d("SQLite", "Se actualizó 'updated_at' para el Aval ID: " + localAval.getId());
                        } else {
                            Log.e("SQLite", "Error al actualizar 'updated_at' para el Aval ID: " + localAval.getId());
                        }
                    }

                    // Comparar fechas para decidir si actualizar
                    if (localUpdatedAt.isEmpty() || isNewer(serverAval.getUpdated_at(), localUpdatedAt)) {
                        Log.d("AvalesWorker", "El aval en el servidor es más nuevo. Actualizando en SQLite...");
                        Log.d("AvalesWorker", "Direccion a actualizar: " + serverAval.getDireccion());

                        updateValuesInDatabase(db, serverAval, remainingComparisons);
                    } else if (isNewer(localUpdatedAt, serverAval.getUpdated_at())) {
                        Log.d("AvalesWorker", "El aval local es más nuevo. Enviando datos al servidor...");
                        enviarDatosLocalAlServidor(context, localAval, db, remainingComparisons);
                    } else {
                        Log.d("AvalesWorker", "Los avales son iguales. No se requiere actualización.");
                        remainingComparisons.decrementAndGet();
                    }
                }
            } else {
                Log.d("AvalesWorker", "El aval no existe en la base de datos local. Insertando...");
                insertAvalesIntoSQLite(context, Collections.singletonList(serverAval), db, remainingComparisons);
            }

            // Cerrar la base de datos si todas las comparaciones han finalizado
            if (remainingComparisons.get() == 0) {
                Log.d("AvalesWorker", "Todas las comparaciones finalizadas. Cerrando base de datos.");
                db.close();
                isTaskAvalesCompleted = true;
            }
        }
    }



    private void enviarDatosLocalAlServidor(Context context, Aval avalLocal, SQLiteDatabase db, AtomicInteger remainingComparisons) {
        remainingComparisons.decrementAndGet();
        Handler mainHandler = new Handler(Looper.getMainLooper());

        if (context == null) {
            Log.e("AvalesWorker", "Context is null. Cannot send data to server.");
            return; // Salir si el contexto es nulo
        }

        JSONObject data = new JSONObject();
        try {
            data.put("func", "updateFromLocal");
            data.put("id", avalLocal.getId());  // Suponiendo que tienes un aval local
            data.put("nombre", avalLocal.getNombre_completo());
            data.put("direccion", avalLocal.getDireccion());
            data.put("telefono", avalLocal.getTelefono());
            data.put("otras_referencias", avalLocal.getOtras_referencias());
            data.put("garantias", avalLocal.getGarantias());
            data.put("updated_at", avalLocal.getUpdated_at());
            data.put("ruta_id", avalLocal.getRuta_id());
            data.put("poblacion_id", avalLocal.getPoblacion_id());
            data.put("colocadora_id", avalLocal.getColocadora_id());

            Log.d("Avales", "Parametros enviados al servidor para actualizar AVALES: " + data.toString());
        } catch (JSONException e) {
            e.printStackTrace();
            return;  // Si hay un error en la creación del JSON, salir
        }

        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlAvales, data,
                response -> {
                    //Log.d("AvalesWorker", "Datos locales enviados y respuesta recibida: " + response.toString());
                },
                error -> {
                    mainHandler.post(() -> {
                        String errorMessage = error.getMessage();
                        Log.e("AvalesWorker", "Error occurred al enviar datos locales AVALES: " + errorMessage, error);
                        // Comprobar el tipo de error (puede ser un error de red, tiempo de espera, etc.)
                        if (error.networkResponse != null) {
                            Log.e("AvalesWorker", "Código de estado: " + error.networkResponse.statusCode);
                        }
                    });
                }
        );

        request.setRetryPolicy(new DefaultRetryPolicy(
                10000, // 30 segundos de timeout
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));

        VolleySingleton.getInstance(context.getApplicationContext()).addToRequestQueue(request);
    }

    private void insertAvalesIntoSQLite(Context context, List<Aval> avales, SQLiteDatabase db, AtomicInteger remainingComparisons) {
        Log.e("SQLite", "Entrando a la función insertAvalesIntoSQLite()............................");
        try {
            for (Aval aval : avales) {
                // El aval no existe, insertarlo
                ContentValues values = new ContentValues();
                values.put("id", aval.getId());
                values.put("nombre_completo", aval.getNombre_completo());
                values.put("direccion", aval.getDireccion());
                values.put("telefono", aval.getTelefono());
                values.put("garantias", aval.getGarantias());
                values.put("carpeta_comprobantes", aval.getCarpeta_comprobantes());
                values.put("carpeta_garantias", aval.getCarpeta_garantias());
                values.put("otras_referencias", aval.getOtras_referencias());
                values.put("created_at", aval.getCreated_at());
                values.put("updated_at", aval.getUpdated_at());
                values.put("update_comprobantes", aval.getUpdate_comprobantes_Enaval());
                values.put("update_garantias", aval.getUpdate_garantias_Enaval());
                values.put("ruta_id", aval.getRuta_id() != null ? aval.getRuta_id() : "NULL");
                values.put("poblacion_id", aval.getPoblacion_id() != null ? aval.getPoblacion_id() : "NULL");
                values.put("colocadora_id", aval.getColocadora_id() != null ? aval.getColocadora_id() : "NULL");

                long newRowId = db.insertWithOnConflict("avales", null, values, SQLiteDatabase.CONFLICT_REPLACE);
                if (newRowId == -1) {
                    remainingComparisons.decrementAndGet();
                    Log.e("SQLite", "Error al insertar el aval con ID: " + aval.getId());
                }
                else
                {
                    remainingComparisons.decrementAndGet();
                }
            }
        } catch (Exception e) {
            Log.e("SQLite", "Error durante la inserción en lote: " + e.getMessage());
        }
    }

    private void updateValuesInDatabase(SQLiteDatabase db, Aval aval, AtomicInteger remainingComparisons) {
        ContentValues values = new ContentValues();
        values.put("nombre_completo", aval.getNombre_completo());
        values.put("direccion", aval.getDireccion());
        values.put("telefono", aval.getTelefono());
        values.put("garantias", aval.getGarantias());
        values.put("carpeta_comprobantes", aval.getCarpeta_comprobantes());
        values.put("carpeta_garantias", aval.getCarpeta_garantias());
        values.put("otras_referencias", aval.getOtras_referencias());
        values.put("created_at", aval.getCreated_at());
        values.put("updated_at", aval.getUpdated_at());
        values.put("update_comprobantes", aval.getUpdate_comprobantes_Enaval());
        values.put("update_garantias", aval.getUpdate_garantias_Enaval());
        values.put("ruta_id", aval.getRuta_id() != null ? aval.getRuta_id() : "NULL");
        values.put("poblacion_id", aval.getPoblacion_id() != null ? aval.getPoblacion_id() : "NULL");
        values.put("colocadora_id", aval.getColocadora_id() != null ? aval.getColocadora_id() : "NULL");

        // Actualizar el registro en la base de datos
        int rowsUpdated = db.update("avales", values, "id=?", new String[]{String.valueOf(aval.getId())});
        if (rowsUpdated > 0) {
            remainingComparisons.decrementAndGet();
            Log.d("SQLite", "Registro actualizado correctamente: ID = " + aval.getId());
        } else {
            remainingComparisons.decrementAndGet();
            Log.e("SQLite", "Error al actualizar el registro. ID no encontrado o fallo en la base de datos.");
        }
    }

    private boolean isNewer(String serverDate, String localDate) {
        // Verificar si alguna de las fechas es nula o vacía
        if (serverDate == null || localDate == null || serverDate.isEmpty() || localDate.isEmpty()) {
            Log.e("isNewer", "Una de las fechas es nula o vacía: serverDate=" + serverDate + ", localDate=" + localDate);
            return false; // O decide cómo manejar este caso
        }

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", new Locale("es", "MX"));
        try {
            Date server = sdf.parse(serverDate);
            Date local = sdf.parse(localDate);

            return server != null && local != null && server.after(local);
        } catch (ParseException e) {
            e.printStackTrace();
            return false; // En caso de error, asumir que no es más reciente
        }
    }

    private boolean isNetworkAvailable(Context context) {
        try {
            NetworkInfo activeNetworkInfo = connectivityManager.getActiveNetworkInfo();
            return activeNetworkInfo != null && activeNetworkInfo.isConnected();
        } catch (Exception e) {
            Log.e("AvalesWorker", "Error checking network availability", e);
            return false;
        }
    }

    public static void cancelWork(Context context) {
        WorkManager.getInstance(context).cancelUniqueWork("AvalesWorker");
        Log.d("AVALES", "Worker cancelado");
    }
}
